*******************************************************
* Load data
*******************************************************

set type double

use "$inputs/hhis_lambdas", clear

// Adjust lambdas for share of private companies in the economy
gen private_share = 0.587

gen lambda_inter_ = lambda_inter*(1-private_share)^2
gen lambda_intra_ = lambda_intra*(1-private_share)^2

gen mhhi = hhi + lambda_intra_*(1-hhi)
gen mhhi_labor = hhi_labor + lambda_intra_*(1-hhi_labor)
gen mhhi_capital = hhi_capital+lambda_inter_*(1-hhi_capital)

keep if fyear>=1985 & fyear<=2015

// Define elasticity parameters

gen alpha = 2/3 
gen eta = 0.59 // Chetty et al.
gen nsectors = 453
gen theta = 3/(1-1/nsectors) // Nechio and Hobijn
gen sigma = 1/2 // Gruber
gen KL = 0.218672736681127 // Fred capital per worker in 1985, in millions of 2011 USD
gen KY = 3.22189239272912 // Fred capital output ratio in 1985
gen epop = 0.6 // EPOP in 1985

gen delta = 0.1 
gen beta = 0.99 
gen r_star = 1.071 // Real interest rate in 1985
gen s_star = r_star^((1-sigma)/sigma)*beta^(1/sigma) / (1+r_star^((1-sigma)/sigma)*beta^(1/sigma))
gen elast_k_cal = ((1-sigma)/sigma) * (1-s_star) * (1-(1-delta)/r_star)
gen rrr_star = r_star-1+delta

*******************************************************
* Calculate labor markdown and labor share
*******************************************************

gen mu_labor = mhhi_labor/eta
gen mu_product = (mhhi-lambda_inter_)/theta
gen one_plus_mu_l = (1+mu_labor)/(1-mu_product)

gen mu_capital_cal = mhhi_capital/elast_k_cal
gen one_plus_mu_k_cal = (1+mu_capital_cal)/(1-mu_product)

*******************************************************
* Calibrate endowment and productivity
*******************************************************

gen k_star = KL
gen endowment = k_star/s_star

gen A_ = one_plus_mu_k_cal*rrr_star / ((1-alpha)*(k_star/epop)^(-alpha))
egen A = sum(A_*(fyear==1985))
drop A_

gen omega_ = A*alpha*(k_star/epop)^(1-alpha) / one_plus_mu_l
egen omega = total(omega_*(fyear==1985))
drop omega_
gen chi = omega / (epop)^(1/eta)

save "$path_temp/calibrated_parameters", replace

****************************************************************************
* Calculate capital markdown and capital share -- full model
****************************************************************************

use "$path_temp/calibrated_parameters", clear

expand 100000
sort fyear
by fyear: gen s = _n/100000

gen k = s*endowment
gen r = (s/(1-s))^(sigma/(1-sigma))*(1/beta)^(1/(1-sigma))
gen elast_k = ((1-sigma)/sigma) * (1-s) * (1-(1-delta)/r)

gen mu_capital = mhhi_capital/elast_k
gen one_plus_mu_k = (1+mu_capital)/(1-mu_product)

gen l = (A*alpha*k^(1-alpha)/(chi*one_plus_mu_l))^(1/(1-alpha+1/eta))

gen mpk = A*(1-alpha)*(k/l)^(-alpha)

gen obj_fun = abs(mpk / (one_plus_mu_k*(r-1+delta))-1)

by fyear: egen min_obj = min(obj_fun)
keep if obj_fun==min_obj

by fyear: gen N = _N
assert N==1
drop N

keep fyear k elast_k mu_capital one_plus_mu_k one_plus_mu_l alpha r s endowment mpk l A alpha

gen capital_share = (1-alpha)/one_plus_mu_k
gen labor_share = alpha/one_plus_mu_l
list fyear capital_share r  elast_k mpk k s l labor_share

keep fyear capital_share labor_share r mpk
rename (capital_share labor_share) (capital_share_model labor_share_model)
save "$path_temp/capital_share_model", replace

****************************************************************
* Calculate capital markdown and capital share -- no CO
****************************************************************

use "$path_temp/calibrated_parameters", clear

drop one_plus_mu_l mu_labor mu_product
gen mu_labor = hhi_labor/eta
gen mu_product = hhi/theta
gen one_plus_mu_l = (1+mu_labor)/(1-mu_product)

expand 10000
sort fyear
by fyear: gen s = _n/100000

gen k = s*endowment
gen r = (s/(1-s))^(sigma/(1-sigma))*(1/beta)^(1/(1-sigma))
gen elast_k = ((1-sigma)/sigma) * (1-s) * (1-(1-delta)/r)

gen mu_capital = hhi_capital/elast_k
gen one_plus_mu_k = (1+mu_capital)/(1-mu_product)

gen l = (A*alpha*k^(1-alpha)/(chi*one_plus_mu_l))^(1/(1-alpha+1/eta))

gen mpk = A*(1-alpha)*(k/l)^(-alpha)

gen obj_fun = (mpk / (one_plus_mu_k*(r-1+delta))-1)^2

by fyear: egen min_obj = min(obj_fun)
keep if obj_fun==min_obj

by fyear: gen N = _N
assert N==1
drop N

keep fyear k elast_k mu_capital one_plus_mu_k one_plus_mu_l alpha r s endowment mpk l A alpha

gen capital_share = (1-alpha)/one_plus_mu_k
gen labor_share = alpha/one_plus_mu_l
list fyear capital_share r  elast_k mpk k s l labor_share

keep fyear capital_share labor_share
rename (capital_share labor_share) (capital_share_noco labor_share_noco)
save "$path_temp/capital_share_noco", replace


****************************************************************************************
* Calculate capital markdown and capital share -- only oligopoly, no common ownership
****************************************************************************************

use "$path_temp/calibrated_parameters", clear

gen mu_neary = hhi/theta
gen one_plus_mu_neary = 1/(1-mu_neary)
gen labor_share_neary =alpha*(1-mu_neary)
gen capital_share_neary =(1-alpha)*(1-mu_neary)

keep fyear capital_share_neary labor_share_neary
save "$path_temp/capital_share_neary", replace

****************************************************************************************
* Combine results and plot
****************************************************************************************

merge 1:1 fyear using "$path_temp/capital_share_noco"
drop _m
merge 1:1 fyear using "$path_temp/capital_share_model"
drop _m
merge 1:1 fyear using "$inputs/income_shares_kn"
drop _m
gen capital_share = sK_it5 + sK_nit5
merge 1:1 fyear using "$inputs/labor_share_bls.dta"
drop _m
replace labor_share = labor_share/100

tsset fyear
sort fyear

// Filter actual values

tsfilter hp labor_share_cycle=labor_share, trend(labor_share_trend) smooth(100)
drop labor_share_cycle

tsfilter hp capital_share_cycle=capital_share, trend(capital_share_trend) smooth(100)
drop capital_share_cycle

// Filter labor share calibration

tsfilter hp labor_share_cycle=labor_share_model, trend(labor_share_model_) smooth(100)
drop labor_share_model labor_share_cycle
rename labor_share_model_ labor_share_model

tsfilter hp labor_share_cycle=labor_share_neary, trend(labor_share_neary_) smooth(100)
drop labor_share_neary labor_share_cycle
rename labor_share_neary_ labor_share_neary

tsfilter hp labor_share_cycle=labor_share_noco, trend(labor_share_noco_) smooth(100)
drop labor_share_noco labor_share_cycle
rename labor_share_noco_ labor_share_noco

// Filter capital share calibration

tsfilter hp capital_share_cycle=capital_share_model, trend(capital_share_model_) smooth(100)
drop capital_share_model capital_share_cycle
rename capital_share_model_ capital_share_model

tsfilter hp capital_share_cycle=capital_share_neary, trend(capital_share_neary_) smooth(100)
drop capital_share_neary capital_share_cycle
rename capital_share_neary_ capital_share_neary

tsfilter hp capital_share_cycle=capital_share_noco, trend(capital_share_noco_) smooth(100)
drop capital_share_noco capital_share_cycle
rename capital_share_noco_ capital_share_noco

list fyear capital_share_trend capital_share_model if fyear==1985
list fyear labor_share_trend labor_share_model if fyear==1985

// Normalize variables to equal 1 in 1985

egen x = sum(labor_share_trend*(fyear==1985))
replace labor_share_trend = labor_share_trend/x
drop x
egen x = sum(labor_share_model*(fyear==1985))
replace labor_share_model = labor_share_model/x
drop x
egen x = sum(labor_share_neary*(fyear==1985))
replace labor_share_neary = labor_share_neary/x
drop x
egen x = sum(labor_share_noco*(fyear==1985))
replace labor_share_noco = labor_share_noco/x
drop x

egen x = sum(capital_share_trend*(fyear==1985))
replace capital_share_trend = capital_share_trend/x
drop x
egen x = sum(capital_share_model*(fyear==1985))
replace capital_share_model = capital_share_model/x
drop x
egen x = sum(capital_share_neary*(fyear==1985))
replace capital_share_neary = capital_share_neary/x
drop x
egen x = sum(capital_share_noco*(fyear==1985))
replace capital_share_noco = capital_share_noco/x
drop x

// Label variables


label var labor_share_model "Labor Share (Oligopoly + Oligopsony + CO)"
label var labor_share_neary "Labor Share (Oligopoly)"
label var labor_share_noco "Labor Share (Oligopoly + Oligopsony)"

label var capital_share_model "Capital Share (Oligopoly + Oligopsony + CO)"
label var capital_share_neary "Capital Share (Oligopoly)"
label var capital_share_noco "Capital Share (Oligopoly + Oligopsony)"

label var labor_share_trend "Labor Share (BLS)"
label var capital_share_trend "Capital Share (Karabarbounis and Neiman)"
label var fyear "Year"

keep if fyear>=1985 & fyear<=2015

// Plot labor shares

twoway  (line labor_share_trend fyear, lcolor(black) lpat(dash)) (scatter labor_share_neary fyear, lcolor(midgreen) msymbol(diamond) mcolor(midgreen) connect(direct)) (scatter labor_share_noco fyear, lcolor(emerald) mcolor(emerald) msymbol(square) connect(direct)) (scatter labor_share_model fyear, lcolor(midblue) mcolor(midblue) msymbol(circle) connect(direct)) if fyear>=1985, legend(size(vsmall))  ylabel(0.9(0.05)1.05, grid) xlabel(1985(5)2015, grid)
graph export "$output/labor_share_calibration_base.pdf", replace

// Plot capital shares

twoway  (line capital_share_trend fyear, lcolor(black) lpat(dash)) (scatter capital_share_neary fyear, lcolor(midgreen) msymbol(diamond) mcolor(midgreen) connect(direct)) (scatter capital_share_noco fyear, lcolor(emerald) mcolor(emerald) msymbol(square) connect(direct)) (scatter capital_share_model fyear, lcolor(midblue) mcolor(midblue) msymbol(circle) connect(direct)) if fyear>=1985, legend(size(vsmall))  ylabel(0.3(0.1)1, grid) xlabel(1985(5)2015, grid)
graph export "$output/capital_share_calibration_base.pdf", replace
